import { useState } from 'react'
import { useNavigate } from 'react-router-dom'
import { Cat, Dog, Bird, Rat, Calendar, Clock, User, MessageSquare } from 'lucide-react'
import './BookVisit.css'

const animals = [
  { id: 'cat', name: 'Кот', icon: Cat },
  { id: 'dog', name: 'Собака', icon: Dog },
  { id: 'bird', name: 'Птица', icon: Bird },
  { id: 'rodent', name: 'Грызун', icon: Rat }
]

function BookVisit() {
  const navigate = useNavigate()
  const [selectedAnimal, setSelectedAnimal] = useState(null)
  const [selectedDate, setSelectedDate] = useState(null)
  const [selectedTime, setSelectedTime] = useState(null)
  const [petName, setPetName] = useState('')
  const [ownerName, setOwnerName] = useState('')
  const [phone, setPhone] = useState('')
  const [email, setEmail] = useState('')
  const [reason, setReason] = useState('')

  const handleSubmit = (e) => {
    e.preventDefault()
    
    if (!selectedAnimal || !selectedDate || !selectedTime || !petName || !ownerName || !phone) {
      alert('Пожалуйста, заполните все обязательные поля')
      return
    }

    const visits = JSON.parse(localStorage.getItem('vetVisits') || '{"active": [], "past": []}')
    const animalName = animals.find(a => a.id === selectedAnimal)?.name || selectedAnimal

    const newVisit = {
      id: Date.now(),
      date: selectedDate,
      time: selectedTime,
      clinic: 'ВетКлиника "Здоровье"',
      diagnosis: reason || 'Запланирован визит',
      animal: animalName,
      petName: petName,
      ownerName: ownerName,
      phone: phone,
      email: email,
      status: 'active'
    }

    visits.active.push(newVisit)
    localStorage.setItem('vetVisits', JSON.stringify(visits))
    
    alert('Запись успешно создана!')
    navigate('/')
  }

  const generateDates = () => {
    const dates = []
    for (let i = 0; i < 14; i++) {
      const date = new Date()
      date.setDate(date.getDate() + i)
      dates.push(date.toISOString().split('T')[0])
    }
    return dates
  }

  const timeSlots = [
    '09:00', '10:00', '11:00', '12:00', '13:00', '14:00', 
    '15:00', '16:00', '17:00', '18:00', '19:00', '20:00'
  ]

  return (
    <div className="book-visit">
      <div className="page-header">
        <h1>Записаться на прием</h1>
        <p>Выберите животное и удобную дату</p>
      </div>

      <form className="booking-form" onSubmit={handleSubmit}>
        <div className="form-section">
          <h2 className="section-title">
            <Cat className="section-icon" size={24} />
            Выберите животное
          </h2>
          <div className="animals-grid">
            {animals.map(animal => {
              const IconComponent = animal.icon
              return (
                <button
                  key={animal.id}
                  type="button"
                  className={`animal-card ${selectedAnimal === animal.id ? 'selected' : ''}`}
                  onClick={() => setSelectedAnimal(animal.id)}
                >
                  <div className="animal-icon">
                    <IconComponent className="animal-icon-img" size={48} />
                  </div>
                  <span className="animal-name">{animal.name}</span>
                </button>
              )
            })}
          </div>
        </div>

        <div className="form-section">
          <h2 className="section-title">
            <Calendar className="section-icon" size={24} />
            Выберите дату визита
          </h2>
          <div className="dates-grid">
            {generateDates().map(date => {
              const dateObj = new Date(date)
              return (
                <button
                  key={date}
                  type="button"
                  className={`date-card ${selectedDate === date ? 'selected' : ''}`}
                  onClick={() => setSelectedDate(date)}
                >
                  <div className="date-day">{dateObj.getDate()}</div>
                  <div className="date-month">{dateObj.toLocaleDateString('ru-RU', { month: 'short' })}</div>
                </button>
              )
            })}
          </div>
        </div>

        <div className="form-section">
          <h2 className="section-title">
            <Clock className="section-icon" size={24} />
            Выберите время
          </h2>
          <div className="times-grid">
            {timeSlots.map(time => (
              <button
                key={time}
                type="button"
                className={`time-card ${selectedTime === time ? 'selected' : ''}`}
                onClick={() => setSelectedTime(time)}
              >
                {time}
              </button>
            ))}
          </div>
        </div>

        <div className="form-section">
          <h2 className="section-title">
            <User className="section-icon" size={24} />
            Информация о питомце и владельце
          </h2>
          <div className="form-fields">
            <div className="form-group">
              <label htmlFor="petName">Имя питомца *</label>
              <input
                type="text"
                id="petName"
                className="form-input"
                placeholder="Введите имя питомца"
                value={petName}
                onChange={(e) => setPetName(e.target.value)}
                required
              />
            </div>
            <div className="form-group">
              <label htmlFor="ownerName">Ваше имя *</label>
              <input
                type="text"
                id="ownerName"
                className="form-input"
                placeholder="Введите ваше имя"
                value={ownerName}
                onChange={(e) => setOwnerName(e.target.value)}
                required
              />
            </div>
            <div className="form-group">
              <label htmlFor="phone">Телефон *</label>
              <input
                type="tel"
                id="phone"
                className="form-input"
                placeholder="+7 (___) ___-__-__"
                value={phone}
                onChange={(e) => setPhone(e.target.value)}
                required
              />
            </div>
            <div className="form-group">
              <label htmlFor="email">Email</label>
              <input
                type="email"
                id="email"
                className="form-input"
                placeholder="your@email.com"
                value={email}
                onChange={(e) => setEmail(e.target.value)}
              />
            </div>
            <div className="form-group full-width">
              <label htmlFor="reason">
                <MessageSquare className="label-icon" size={20} />
                Причина визита
              </label>
              <textarea
                id="reason"
                className="form-textarea"
                placeholder="Опишите причину визита (необязательно)"
                value={reason}
                onChange={(e) => setReason(e.target.value)}
                rows="4"
              />
            </div>
          </div>
        </div>

        <button type="submit" className="submit-btn" disabled={!selectedAnimal || !selectedDate || !selectedTime || !petName || !ownerName || !phone}>
          Записаться на прием
        </button>
      </form>
    </div>
  )
}

export default BookVisit

